/*
      PG_PVM client 
      Author: Edward Walker
      Organization: National Supercomputing Research Centre
      Date: April 1996
*/
/************************************************************************/
/* Include files */
#include <stdio.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include "pvm3.h"
#include "pgtags.h"
#include "pg_pvm.h"

/* useful Macros */
 
#define T_BYTE (sizeof(char))
#define T_CPLX (2*sizeof(float))
#define T_DCPLX (2*sizeof(double))
#define T_DOUBLE (sizeof(double))
#define T_FLOAT (sizeof(float))
#define T_INT (sizeof(int))
#define T_LONG (sizeof(long))
#define T_SHORT (sizeof(short))

#define pg_max(A,B)	(((A) > (B)) ? (A) : (B) )
#define pg_abs(A)	(((A) < 0 ) ? (-A) : (A) )

typedef struct task_node{
     char *label;
     int task;
     struct task_node *next; 
}task_node;

/************************************************************************/
/* Static variables */
static int me=-1;		/* My process number			*/
static int mytid=-1;		/* My tid				*/
static FILE *logfile=stderr;	/* Log file name			*/
static struct timeval ref_time;	/* Reference timestamp			*/
static struct timeval tt;	/* Temp time variable			*/
static struct timeval tt1;	/* Temp time variable			*/
static int initialized=0;	/* Set once pg_tids is called		*/
static char logname[MAXFLTH];
static int called_exit=0;
static int PG_HOME_TID ;
static int me2tid[MAXP];
static int u_rbuf ;
static int u_sbuf ;
static char *pvm_root;
static char *pvm_arch ;
static struct pvmtaskinfo *taskp;
static struct task_node *TaskList;


static int 
pg_tid2me(tid)
    int tid;
{
	int i;
   int serv_cmd ;
   int bufid;

   if ( tid <= 0 ) return -1 ;
   for (i=0;i<MAXP;i++) 
     if ( me2tid[i] == tid ) return i;	
   
   u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
   u_rbuf = pvm_setrbuf(0);

   serv_cmd = PgTid2Me ;   
   pvm_pkint(&serv_cmd,1,1);
   pvm_pkint(&tid,1,1);
   pvm_send(PG_HOME_TID,PG_CMDTAG);

   pvm_recv(PG_HOME_TID,PG_CMDTAG);
   pvm_upkint(&i,1,1);
   if ( 0 <= i && i < MAXP ) me2tid[i] = tid ;

   pvm_freebuf(pvm_setsbuf(u_sbuf));
   pvm_freebuf(pvm_setrbuf(u_rbuf));
   return i;
}

static int 
pg_me2tid(int me)
{
int serv_cmd;
int tid;
int bufid;

   if ( me < 0 || me >= MAXP ) return -1 ;
   if ( me2tid[me] > 0 ) return me2tid[me] ;

   u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
   u_rbuf = pvm_setrbuf(0);


   serv_cmd = PgMe2Tid ;
   pvm_pkint(&serv_cmd,1,1);
   pvm_pkint(&me,1,1);
   pvm_send(PG_HOME_TID,PG_CMDTAG);

   pvm_recv(PG_HOME_TID,PG_CMDTAG);
   pvm_upkint(&tid,1,1);
   if ( tid > 0 ) me2tid[me] = tid ;

   pvm_freebuf(pvm_setsbuf(u_sbuf));
   pvm_freebuf(pvm_setrbuf(u_rbuf));
   return tid;
}

static int 
startup_daemon( void )
{
int mytid, oldval;
int tid[1];
int info ;

   oldval = pvm_setopt(PvmOutputTid,0);
   info = pvm_spawn("pg_serv",0,PvmTaskDefault,0,1,tid);
   pvm_setopt(PvmOutputTid,oldval);
   if ( info != 1 ){
       fprintf(stderr,"Error: pg_serv cannot be started !\n");
       return 0;
   }
   return tid[0] ;
}

static int 
pg_reg_serv( void )
/* PG_HOME_TID must have been set */
{
int serv_cmd, inum;
int bufid;

     u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
     u_rbuf = pvm_setrbuf(0);

     serv_cmd = PgRegister ;
     pvm_pkint(&serv_cmd,1,1);
     pvm_send(PG_HOME_TID,PG_CMDTAG);          
          
     pvm_recv(PG_HOME_TID,PG_CMDTAG);
     pvm_upkint(&inum,1,1);

     pvm_freebuf(pvm_setsbuf(u_sbuf));
     pvm_freebuf(pvm_setrbuf(u_rbuf));
     return inum ;
 
}

static char 
*strip_path( char *prg )
{
char *tmp;
char c;
int i,n;
 
   tmp = (char *)calloc(MAXFLTH,sizeof(char));
   for ( i=0,n=0 ; prg[n] != '\0'; n++ ){
       if ( prg[n] == '/' ){
         i = 0;
         continue ;
       }
       tmp[i] = prg[n];
       ++i;
   }
   tmp[i] = '\0' ;
   return tmp;
}

static int 
pg_register2( void )
/* me and PG_HOME_TID will be set in this routine */
/* assumes mytid has been set */
{
int i,np;
char *prg;

   pvm_tasks(0,&np,&taskp);
   for ( i=0; i<np; i++ ){
      prg = strip_path(taskp[i].ti_a_out); 
      if ( strcmp(prg,"pg_serv") == 0 ){
          PG_HOME_TID = taskp[i].ti_tid ;
          me = pg_reg_serv() ;
          me2tid[me] = mytid ;
          return me;
      }
      free(prg);
   }   
   /* pg_serv has not been started */
   /*
   PG_HOME_TID = startup_daemon() ;
   me = pg_reg_serv();
   me2tid[me] = mytid;
   return me;
   */
   return -1;
}

static void 
timestamp(struct timeval *t)
{
	struct timeval tx;
	double m1, m2;
	gettimeofday(&tx,NULL);
	if (tx.tv_usec >= ref_time.tv_usec) {
	    t->tv_usec = tx.tv_usec-ref_time.tv_usec;
	    t->tv_sec = tx.tv_sec-ref_time.tv_sec;
    }
	else {
	   /* borrow a second */
	    t->tv_usec = (tx.tv_usec+1000000)-ref_time.tv_usec;
	    t->tv_sec = (tx.tv_sec-1)-ref_time.tv_sec;
	}

}

static void 
pg_add_ref( struct timeval *ref, struct timeval delta )
/* ref = ref + delta */
{
struct timeval tmp ;

	tmp.tv_sec = tmp.tv_usec = 0;
	/* 1 millisecond */
	/*
	tmp.tv_sec = 0;
	tmp.tv_usec = 1000;
	*/
	
   if ( delta.tv_sec < 0 ){
		/* delta is -ve */
		printf("%d: delta -ve %d:%d\n", me, delta.tv_sec, delta.tv_usec );
		delta.tv_usec = 1000000 - delta.tv_usec ;
		delta.tv_sec = pg_abs(delta.tv_sec) - 1;
		/* ref = ref - delta */
      if ( ref->tv_usec < delta.tv_usec ){
         ref->tv_usec = 1000000 + ref->tv_usec - delta.tv_usec ;
         ref->tv_sec = ref->tv_sec - 1  - delta.tv_sec ;
      }else{
         ref->tv_sec = ref->tv_sec - delta.tv_sec ;
         ref->tv_usec = ref->tv_usec - delta.tv_usec ;
      }

	}else{
		/* delta is +ve */
		printf("%d: delta +ve %d:%d\n", me, delta.tv_sec, delta.tv_usec );
      /* ref = ref + delta */
      if ( ref->tv_usec + delta.tv_usec < 1000000 ){
         ref->tv_sec = ref->tv_sec + delta.tv_sec ;
         ref->tv_usec = ref->tv_usec + delta.tv_usec ;
      }else{
         ref->tv_usec = (ref->tv_usec + delta.tv_usec) - 1000000 ;
         ref->tv_sec = ref->tv_sec + delta.tv_sec + 1 ;
      }

	}
}

static int 
pg_gettime( int *sec, int *usec )
{
int serv_cmd;
int bufid ;

   u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
   u_rbuf = pvm_setrbuf(0);

   serv_cmd = PgTimeWhat ;
   pvm_pkint(&serv_cmd, 1, 1 );
   pvm_send( PG_HOME_TID, PG_CMDTAG);
 
   pvm_recv( PG_HOME_TID, PG_CMDTAG); 
   pvm_upkint( sec, 1, 1 );
   pvm_upkint( usec, 1, 1 );

   pvm_freebuf(pvm_setsbuf(u_sbuf));
   pvm_freebuf(pvm_setrbuf(u_rbuf));
}

static struct timeval 
pg_clksync( void )
{
struct timeval tmp;
int usec, sec;

	tmp.tv_sec = tmp.tv_usec = 0; 
	if (initialized){
      pg_gettime(&sec,&usec);
      ref_time.tv_sec = sec;
      ref_time.tv_usec = usec ;
   	if ( me > 0 ){
			timestamp(&tmp);
		}
	}
	return tmp;
}


/************************************************************************/
static int 
init_stuff( void )
{
int i;

   for(i=0;i<MAXP;i++) me2tid[i] = 0;
   TaskList = NULL;
	mytid = pvm_mytid();
}

static int 
pg_tids( void )
{
	char hostname[100];
	register int i;
	FILE *fd;
	char new_export[100], *pvm_export ;

	if (!initialized) {
	  initialized=1;
     init_stuff(); 
	  if (mytid <= 0)
		return (0);
	  /* register process into profile group */
	  if ( pg_register2() < 0 ){
		 fprintf(stderr,"Error: %x did not register successfully!\n", mytid );
       pvm_exit();
	    exit(0);
     }
	  tt = pg_clksync() ;
	  /* gethostname(hostname,sizeof(hostname)); */
	  sprintf(logname, "/tmp/pgfile.%d.%d",getuid(),getpid());
	  fd = fopen (logname,  "w+");
	  if (fd!=0) {
			logfile=fd;
	    	setvbuf(logfile, NULL, _IOFBF, 10240);
	  }else {
			printf("pg_tids:unable to open logfile\n");
     }	

     /* PG: trace_start */
	  fprintf (logfile, "-3 -901 %d.%06d %d -1 0\n",tt.tv_sec, tt.tv_usec, me);
	  /* printf("%d: pg_tids: initialization completed  \n", me ); */
	}
	return PvmOk;
}

/************************************************************************/

static struct task_node 
*NewTaskNode( char *label, int task )
{
struct task_node *tmp;

   tmp = (struct task_node *)calloc(1,sizeof(struct task_node));
   tmp->task = task ;
   tmp->label = (char *)calloc(strlen(label)+1,sizeof(char));
   strcpy(tmp->label,label);
   tmp->next = NULL;
   return tmp;
}

static int
pg_get_task( char *label )
{
int task;
int serv_cmd;
struct task_node *tmp;
int bufid ;
 
   u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
   u_rbuf = pvm_setrbuf(0);

   serv_cmd = PgTaskLabel ;
   pvm_pkint(&serv_cmd,1,1);
   pvm_pkstr(label);
   pvm_send(PG_HOME_TID,PG_CMDTAG);

   /* wait for reply */
   pvm_recv(PG_HOME_TID,PG_CMDTAG);
   pvm_upkint(&task,1,1);

   tmp = NewTaskNode(label,task); 
   tmp->next = TaskList ;
   TaskList = tmp;

   pvm_freebuf(pvm_setsbuf(u_sbuf));
   pvm_freebuf(pvm_setrbuf(u_rbuf));
   return task;
}

int
pg_start( char *label )
{
int task;
struct timeval tt;
struct task_node *tmp;

if ( !initialized ) pg_tids();
task = -1;
for(tmp=TaskList; tmp!=NULL; tmp=tmp->next){
   if ( strcmp(tmp->label,label) == 0 ){
       task = tmp->task ;
       break ;
   }
}
if ( task == -1 ) task = pg_get_task(label);
if ( task == -1 ) return PvmNoTask;

timestamp(&tt);
/* PG : BLOCK_BEGIN */
fprintf (logfile, "-3 %d %d.%06d %d -1 0\n",
        task, tt.tv_sec, tt.tv_usec, me );
/*
printf ("-3 %d %d.%06d %d -1 0\n",
        task, tt.tv_sec, tt.tv_usec, me );
*/

return PvmOk;
}

int 
pg_end( char *label )
{
int task ;
struct timeval tt1;
struct task_node *tmp;

if ( !initialized ) return PvmNoTask;

task = -1 ;
for(tmp=TaskList; tmp!=NULL; tmp=tmp->next){
   if ( strcmp(tmp->label,label) == 0 ){
       task = tmp->task ;
       break ;
   }
}
if ( task == -1 ) return PvmNoTask;

timestamp(&tt1);
/* PG : BLOCK_END */
fprintf (logfile, "-4 %d %d.%06d %d -1 0\n",
     task, tt1.tv_sec, tt1.tv_usec, me );
/*
printf ("-4 %d %d.%06d %d -1 0\n",
     task, tt1.tv_sec, tt1.tv_usec, me );
*/

return PvmOk;
}
/**********************************************************************/

static int datasize(dtype)
     int dtype;
{

switch(dtype) {
case PVM_STR:
     return(T_BYTE);
case PVM_BYTE:
     return(T_BYTE);
case PVM_USHORT:
case PVM_SHORT:
     return(T_SHORT);
case PVM_UINT:
case PVM_INT:
     return(T_INT);
case PVM_ULONG:
case PVM_LONG:
     return(T_LONG);
case PVM_FLOAT:
     return(T_FLOAT);
case PVM_DOUBLE:
     return(T_DOUBLE);
case PVM_CPLX:
     return(T_CPLX);
case PVM_DCPLX:
     return(T_DCPLX);
}
return 0;
}

/*****************************************************************/
static void
cp2serv( char *infile )
{
char *buf ;
int n;
int tcnt,lcnt, pkcnt;
char line[MAXLTH];
FILE *fp;
 
    if ((fp=fopen(infile,"r")) == NULL){
        fprintf(stderr,"Warning: cannot open %s \n", infile );
    }else{

       u_sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
       u_rbuf = pvm_setrbuf(0);

       pkcnt = 0;
       buf = (char *)calloc(TRACEBUFSIZ,sizeof(char));
       tcnt = 0;
       buf[0] = '\0' ;
       while( fgets(line,MAXLTH,fp) != NULL ){
          lcnt = strlen(line);
          if ( tcnt + lcnt < TRACEBUFSIZ ){
             buf = strcat(buf,line);
             tcnt = tcnt + lcnt ;
          }else{
             pvm_initsend(PvmDataDefault);
             n = -1;
             pvm_pkint(&n,1,1);
             pvm_pkstr(buf); 
             pvm_send(PG_HOME_TID, PG_TRACETAG);
             ++pkcnt;
             tcnt = 0;
             buf[0] = '\0' ;
             buf = strcat(buf,line);
             tcnt = tcnt + lcnt ;
          }
       }
       lcnt = strlen(buf);
       if ( lcnt > 0 ){
          pvm_initsend(PvmDataDefault);
          n = -1;
          pvm_pkint(&n,1,1);
          pvm_pkstr(buf); 
          pvm_send(PG_HOME_TID, PG_TRACETAG);
          ++pkcnt;
       }
       pvm_initsend(PvmDataDefault);
       pvm_pkint(&pkcnt,1,1);
       pvm_send(PG_HOME_TID, PG_TRACETAG);

       /* wait for acknowledgement */
       pvm_recv(PG_HOME_TID,PG_TRACETAG);
       fclose(fp);
       free(buf);

       pvm_freebuf(pvm_setsbuf(u_sbuf));
       pvm_freebuf(pvm_setrbuf(u_rbuf));
   }
}

int pg_exit()
{
int tid ;
int inum,retval;
int f1,f2,n,i,grpid;
char buf[TRACEBUFSIZ];
int info ;
char outfile[100] ;
static char *pwd ;
char trf_file[100];

if (called_exit){
	return pvm_exit();
}
called_exit=1;
if(initialized) {
   timestamp(&tt);
   /* PG: trace_end */
   fprintf (logfile, "-4 -901 %d.%06d %d -1 0\n", tt.tv_sec, tt.tv_usec, me);
   fclose(logfile);
   /* send my tracefile to the server process */
   /* printf("sending trace file to server %d \n", PG_HOME_TID );*/
   if ( PG_HOME_TID > 0 ){
      cp2serv(logname);
      unlink(logname); 
   }
   /* printf("%d:calling pvm_exit ... \n", me ); */
   info = pvm_exit();
   /* printf("%d:pvm_exit = %d! \n", me, info ); */
}else{
   info = pvm_exit();
}
return info ;
}
/************************************************************************/

int pg_close()
{
	int tid ;
   int inum,retval;
	int f1,f2,n,i;
	char buf[TRACEBUFSIZ];
	if (called_exit) return;
	called_exit=1;
	if(initialized) {
	  	timestamp(&tt);
		/* PG: trace_end */
	  	fprintf (logfile, "-4 -901 %d.%06d %d -1 0\n", tt.tv_sec, tt.tv_usec, me);
	  	fclose(logfile);
		/* send my tracefile to the home process */
      if ( PG_HOME_TID > 0 ){
         cp2serv(logname);
         unlink(logname);
      }
   }
  	initialized = 0;
}
/*************************************************************************/
int pg_recv(tid, msgtag)
    int tid;
    int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who;

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
	info = bufid = pvm_recv(tid, msgtag);
	timestamp(&tt1);
	if(bufid > 0) {
	   (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
	   who = pg_tid2me(rid);
		if (who >= 0) {
			/* printf("%d -<recv>-> %d \n", who, me ); */
         /* PG: recv_blocking */
	      fprintf (logfile, "-3 -52 %d.%06d %d -1 1 2 %d\n", 
		      	tt.tv_sec, tt.tv_usec, me, tag);
         /* PG: recv_waking */
	      fprintf (logfile, "-4 -52 %d.%06d %d -1 3 2 %d %d %d\n",
		     		tt1.tv_sec, tt1.tv_usec, me, nb, tag, who);
      }
	}
	return info;
}
/************************************************************************/
int pg_nrecv(tid, msgtag)
    int tid;
    int msgtag;
{
	int info;
	int nb, rid, tag, bufid, who;

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
	info = bufid = pvm_nrecv(tid, msgtag);
   timestamp(&tt1);
	if(bufid > 0) {
	   (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
	   who = pg_tid2me(rid);
		if (who >= 0) {
        	/* PG: recv_blocking */
	      fprintf (logfile, "-3 -52 %d.%06d %d -1 1 2 %d\n",
               tt.tv_sec, tt.tv_usec, me, tag);
 
         /* PG: recv_waking */
         fprintf (logfile, "-4 -52 %d.%06d %d -1 3 2 %d %d %d\n",
               tt1.tv_sec, tt1.tv_usec, me, nb, tag, who);	
    	}
	}
	return info;
}

/************************************************************************/
int pg_trecv(tid, msgtag, tmout)
    int tid;
    int msgtag;
	struct timeval *tmout;
{
	int info;
	int nb, rid, tag, bufid, who;

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
	info = bufid = pvm_trecv(tid, msgtag, tmout);
	timestamp(&tt1);
	if(bufid >= 0) {
      (void)pvm_bufinfo(bufid, &nb, &tag, &rid);
      who = pg_tid2me(rid);
      if (who >= 0) {
          /* PG: recv_blocking */
          fprintf (logfile, "-3 -52 %d.%06d %d -1 1 2 %d\n",
               tt.tv_sec, tt.tv_usec, me, tag);	
          /* PG: recv_waking */
	       fprintf (logfile, "-4 -52 %d.%06d %d -1 3 2 %d %d %d\n",
               tt1.tv_sec, tt1.tv_usec, me, nb, tag, who);  
      }
	}
	return info;
}

/************************************************************************/
int pg_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt)
    int tid, msgtag,nitems,dtype,*rtid,*rtag,*rcnt;
    void* buf;
{
	int info, nb, who;

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
   info = pvm_precv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt);
	timestamp(&tt1);
   if ( info >= 0 ){
	   who = pg_tid2me(*rtid);
      /* printf("%d -recv-> %d \n", who, me ); */
	   nb = (datasize(dtype))*(*rcnt);
	   if (who >= 0) {
          /* PG: recv_blocking */
          fprintf (logfile, "-3 -52 %d.%06d %d -1 1 2 %d\n",
               tt.tv_sec, tt.tv_usec, me, *rtag);	
		     		
          /* PG: recv_waking */
	       fprintf (logfile, "-4 -52 %d.%06d %d -1 3 2 %d %d %d\n",
               tt1.tv_sec, tt1.tv_usec, me, nb, *rtag, who);  
         /* printf("%d -recv-> %d (tag=%d,size=%d)\n", who,me,*rtag,nb); */
      }
   }
	return info;
}

/************************************************************************/
int pg_mcast (ttids, ntid, msgtag)
    int *ttids, ntid, msgtag;
{
	int i, info, bufid, nb, idum, who,found_one=0;
	int send_inst;
	int s_tag ;
	

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
	info = pvm_mcast(ttids, ntid, msgtag);
	timestamp(&tt1);
	bufid=pvm_getsbuf();
	(void)pvm_bufinfo(bufid, &nb, &idum, &idum);
	/* printf("mcast --> "); */
	for (i=0;i<ntid;i++) {
	    who = pg_tid2me(ttids[i]);
       /* printf(" %d ", who ); */
       if (who >= 0) {
          found_one = 1;
          /* PG: send */
	       fprintf (logfile, "-3 -21 %d.%06d %d -1 3 2 %d %d %d\n",
		     	tt.tv_sec, tt.tv_usec, me, nb, msgtag, who);
       }
	}
	/* printf("\n"); */
	if (found_one) {
      /* PG: send_end */
      fprintf (logfile, "-4 -21 %d.%06d %d -1 0\n",
          tt1.tv_sec, tt1.tv_usec, me );
   }
	return info;
}

/************************************************************************/
int pg_spawn(char *task, char **argv, int flag, char *where, 
int ntask, int *tids )
{
int info ;
int spawn_event;
int gsize, ngsize;
int i, who;

   if ( !initialized ) pg_tids() ;
   timestamp(&tt);
   info = pvm_spawn( task,argv,flag,where,ntask,tids);
   timestamp(&tt1);
   /* PG : spawn_begin */
   /*
   fprintf (logfile, "-3 -501 %d.%06d %d -1 1 2 %d\n",
      tt.tv_sec, tt.tv_usec, me, ntask );
   */
   if ( info >= 0 ){
      spawn_event = info;
   }else{
      spawn_event = 0;
   }
   for(i=0;i<info;i++){
      who = pg_tid2me(tids[i]);
      fprintf (logfile, "-2 -503 %d.%06d %d -1 2 2 %d -1\n",
          tt1.tv_sec, tt1.tv_usec, me, who);
   }
   /* PG: spawn_end */
   /*
   fprintf (logfile, "-4 -501 %d.%06d %d -1 1 2 %d\n",
	   tt1.tv_sec, tt1.tv_usec, me, spawn_event);
   */

   return info ;
}
/************************************************************************/
int pg_send (tid, msgtag)
    int tid, msgtag;
{
	int info, who, bufid, nb, idum1, idum2;
	int send_inst ;
	int s_tag ;

   if ( !initialized ) pg_tids() ; 
	timestamp(&tt);
	info = pvm_send(tid, msgtag);
	timestamp(&tt1);
	bufid=pvm_getsbuf();
	(void)pvm_bufinfo(bufid, &nb, &idum1, &idum2);
	who = pg_tid2me(tid);
	if (who >= 0) {
		/* printf("%d -send(%d)-> %d \n", me, msgtag, who ); */
      /* PG: send */
	   fprintf (logfile, "-3 -21 %d.%06d %d -1 3 2 %d %d %d\n",
            tt.tv_sec, tt.tv_usec, me, nb, msgtag, who); 
      /* PG: send_end */
	   fprintf (logfile, "-4 -21 %d.%06d %d -1 0\n",
          tt1.tv_sec, tt1.tv_usec, me ); 
   }
	return info;
}
/************************************************************************/
int pg_psend (tid, msgtag, buf,  nitems, dtype)
    int tid, msgtag, dtype, nitems;
    void* buf;
{
	int info, who;
	int nb = datasize(dtype)*nitems;
	int send_inst ;
	int s_tag; 

   if ( !initialized ) pg_tids() ;
	timestamp(&tt);
	info = pvm_psend(tid, msgtag, buf, nitems ,dtype);
	timestamp(&tt1);
	who = pg_tid2me(tid);
	if ( who >= 0 ){
      /* PG: send */
	  	fprintf (logfile, "-3 -21 %d.%06d %d -1 3 2 %d %d %d\n",
            tt.tv_sec, tt.tv_usec, me, nb, msgtag, who);
      /* printf("%d -send-> %d (tag=%d,size=%d)\n",me,who,msgtag,nb); */
      /* PG: send_end */
	   fprintf (logfile, "-4 -21 %d.%06d %d -1 0\n", 
          tt1.tv_sec, tt1.tv_usec, me );	
	}
	return info;
}

/************************************************************************/

int 
pg_mytid( void )
{
   if ( !initialized ) pg_tids() ;
   return pvm_mytid() ;
}

int pg_initsend( int encode )
{
   if ( !initialized ) pg_tids() ;
   return pvm_initsend( encode ) ;
}

/************************************************************************/
/*			 FORTRAN INTERFACES				*/
/************************************************************************/
void pgfmytid (tid)
int *tid ;
{
	*tid = pg_mytid();
}

void pgfmytid_ (tid)
int *tid;
{
   *tid = pg_mytid();
}
/************************************************************************/
void pgftids (info)
int *info ;
{
	*info = pg_tids(); 
}

void pgftids_(info)
int *info ;
{
   *info = pg_tids(); 
}
/************************************************************************/
void pgfexit (info)
    int *info;
{
	*info = pg_exit();
}

void pgfexit_ (info)
    int *info;
{
   *info = pg_exit();
}

/************************************************************************/
void pgfclose (info)
    int *info;
{
	pg_close();
}

void pgfclose_ (info)
    int *info;
{
   pg_close();
}

/************************************************************************/
void pgfstart(char *label,int *info)
{
    *info = pg_end(label);
}

void pgfstart_ (char *label,int *info)
{
    *info = pg_end(label);
}
/************************************************************************/
void pgfend(char *label, int *info)
{
    *info = pg_start(label);
}

void pgfend_ (char *label, int *info)
{
    *info = pg_start(label);
}
/************************************************************************/
void pgfrecv (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
	*info = pg_recv(*tid, *msgtag);
}

void pgfrecv_ (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
   *info = pg_recv(*tid, *msgtag);
}
/************************************************************************/
void pgfnrecv (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
	*info = pg_nrecv(*tid, *msgtag);
}

void pgfnrecv_ (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
   *info = pg_nrecv(*tid, *msgtag);
}

/************************************************************************/
void pgftrecv (tid, msgtag, sec, usec, info)
    int *tid, *msgtag, *sec, *usec, *info;
{
    struct timeval temp;
	temp.tv_sec = *sec;
	temp.tv_usec = *usec;
	*info = pg_trecv(*tid, *msgtag, &temp);
}

void pgftrecv_ (tid, msgtag, sec, usec, info)
    int *tid, *msgtag, *sec, *usec, *info;
{
    struct timeval temp;
   temp.tv_sec = *sec;
   temp.tv_usec = *usec;
   *info = pg_trecv(*tid, *msgtag, &temp);
}

/************************************************************************/
void pgfmcast (ntid, ttids, msgtag, info)
    int *ntid, *ttids, *msgtag, *info;
{
	*info = pg_mcast(ttids, *ntid, *msgtag);
}

void pgfmcast_ (ntid, ttids, msgtag, info)
    int *ntid, *ttids, *msgtag, *info;
{
   *info = pg_mcast(ttids, *ntid, *msgtag);
}

/************************************************************************/
void pgfspawn ( task, flag, where, ntask, tids, info )
char *task;
int *flag;
char *where;
int *ntask;
int *tids;
int *info;
{
	printf("pg_spawn(%s, %d, %d, %s, %d, tids) \n", task, 0, *flag, where, *ntask );
	*info = pg_spawn( task, 0, *flag, where, *ntask, tids );
}

void pgfspawn_ ( task, flag, where, ntask, tids, info )
char *task;
int *flag;
char *where;
int *ntask;
int *tids;
int *info;
{
/*
printf("pg_spawn(%s, %d, %d, %s, %d, tids) \n",task,0,*flag,where,*ntask);
*/
    
   *info = pg_spawn( task, 0, *flag, where, *ntask, tids );
}

/************************************************************************/
void pgfsend (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
	*info = pg_send(*tid, *msgtag);
}

void pgfsend_ (tid, msgtag, info)
    int *tid, *msgtag, *info;
{
   *info = pg_send(*tid, *msgtag);
}

/************************************************************************/
void pgfinitsend (type, info)
    int *type, *info;
{
	*info = pg_initsend(*type);
}

void pgfinitsend_ (type, info)
    int *type, *info;
{
   *info = pg_initsend(*type);
}

/************************************************************************/
void pgfpsend(tid, msgtag,  buf, nitems, dtype,  info)
    int* tid;
    int* msgtag;
    void* buf;
    int* nitems;
    int* dtype;
    int* info;
{
	*info = pg_psend(*tid, *msgtag, buf, *nitems, *dtype);
}

void pgfpsend_(tid, msgtag, buf, nitems, dtype, info)
    int* tid;
    int* msgtag;
    void* buf;
    int* nitems;
    int* dtype;
    int* info;
{
   *info = pg_psend(*tid, *msgtag, buf, *nitems, *dtype);
}

/************************************************************************/
void pgfprecv(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt, info)
   int* tid;
   int* msgtag;
   void* buf;
   int* nitems;
   int* dtype;
	int* rtid;
	int* rtag;
	int* rcnt;
   int* info;
{
   *info =  pg_precv(*tid, *msgtag, buf, *nitems, *dtype, rtid, rtag, rcnt);
}

void pgfprecv_(tid, msgtag, buf, nitems, dtype, rtid, rtag, rcnt, info)
   int* tid;
   int* msgtag;
   void* buf;
   int* nitems;
   int* dtype;
   int* rtid;
   int* rtag;
   int* rcnt;
   int* info;
{
   *info =  pg_precv(*tid, *msgtag, buf, *nitems, *dtype, rtid, rtag, rcnt);
}

/************************************************************************/
