/*
		PG_PVM server 
      Author: Edward Walker
      Organization: National Supercomputing Research Centre
      Date: April 1996
*/
/************************************************************/
#include <stdio.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include "pvm3.h"
#include "pgtags.h"
#include "pg_pvm.h"
#include "pg_serv.h"

/***************************/
/* Global static variables */

static struct pvmtaskinfo *taskp;
static struct task_node *TaskList=NULL;
static int PG_SEC=-1;
static int PG_USEC=-1;
static int TASK_COUNTER=0;
static int REG_COUNTER=0;
static int TRACE_COUNTER=0;
static int EXIT_COUNTER=0;
static int me2tid[MAXP] ;
static int PKBOUND[MAXP];
static int PKCNT[MAXP];
static int EXIT[MAXP];
static int TOT ;
static int FINISH ;
static char *pwd ;
static char logname[MAXFLTH] ;
static char logname2[MAXFLTH] ;
static struct tid_node *TidList=NULL;

static int
start_tid( int tid )
{
struct tid_node *stid;

	for (stid=TidList; stid != NULL; stid = stid->next){
      if ( stid->tid == tid ) return 1;
	}
	return 0;
}

static void
add_start_tid( int tid )
{
struct tid_node *tmp;

	tmp = (struct tid_node *)malloc(sizeof(struct tid_node));
	if ( tmp == NULL ){
       fprintf(stderr,"Error: No memory (tid_node) \n");
		 exit(0);
	}
	tmp->next = TidList;
	tmp->tid = tid;
   TidList = tmp;
}

static void
free_TidList( void )
{
struct tid_node *trash;

	while (TidList != NULL){
       trash = TidList;
		 TidList = TidList->next ;
		 free(trash);
	}
	TidList = NULL;
}

static void
free_TaskList( void )
{
struct task_node *trash ;

	while (TaskList != NULL){
      trash = TaskList ;
		TaskList = TaskList->next ;
		free(trash);
	}
   TaskList = NULL;
}

static char
*strip_path( char *prg )
{
char *tmp;
char c;
int i,n;
 
   tmp = (char *)calloc(MAXFLTH,sizeof(char));
   for ( i=0,n=0 ; prg[n] != '\0'; n++ ){
       if ( prg[n] == '/' ){
         i = 0;
         continue ;
       }
       tmp[i] = prg[n];
       ++i;
   }
   tmp[i] = '\0' ;
   return tmp;
}

static int
check_tasks( int begin_flag )
{
int i, np;
int tnp;
int tot=0;
char *prg;
int xpvm_tid;

   pvm_tasks(0,&np,&taskp ); 
   tnp = np;
   if ( begin_flag ){
     printf("tnp = %d \n", tnp );
   }
   for (i=0;i<np;i++){
      prg = strip_path(taskp[i].ti_a_out);
      if ( begin_flag ){
         printf("a_out=%s:tid=%x:ptid=%x:flag=%d\n",
            prg,taskp[i].ti_tid,
            taskp[i].ti_ptid,taskp[i].ti_flag);
      }
      /* tasks which we don't count */
      if ( strcmp(prg,"pvmgs") == 0 ){
                /* pvm group server, ie. pvmgs */
                --tnp;
      }else if ( (taskp[i].ti_flag & 0x100) != 0 ){
                /* this is a scheduler */
                --tnp;
      }else if ( (taskp[i].ti_flag & 0x200) != 0 ){
                /* this is a hoster */
                --tnp;
      }else if ( (taskp[i].ti_flag & 0x400) != 0 ){
                /* this is a tasker */
                --tnp; 
      }else{
         /* TASK */
         if ( begin_flag ){
            add_start_tid(taskp[i].ti_tid);
			}else{
            if ( !start_tid(taskp[i].ti_tid) ){
                tot++;
				}
			}
		}
      free(prg);
  }
  return tot;
}

static int
init_stuff( void )
{
int i;
int mytid;

   mytid = pvm_mytid();
   PG_SEC=-1;
   PG_USEC=-1;
   TASK_COUNTER=0;
   REG_COUNTER=0;
   TRACE_COUNTER=0;
   EXIT_COUNTER=0;
   FINISH = 0;
   for(i=0;i<MAXP;i++){ 
      me2tid[i] = 0 ; 
      PKBOUND[i] = -1 ;
      PKCNT[i] = 0 ;
      EXIT[i] = DEAD;
   }
	free_TaskList();
	free_TidList();
   /* wait until it is the only one */
   /* while((TOT=check_tasks(1)) != 1) ; */
	check_tasks(1);
}

/*******************************************************************/

static int 
pg_compare_time( void *vp, void *vq )
{
struct line_node *p = vp, *q = vq;
double x; 
 x = p->ftime - q->ftime ;
 if ( x == 0.0 ){
   if ( p->line < q->line ){
      return -1;
   }else if ( p->line > q->line ){
      return 1;
   }else{
      return 0;
   }
 }else if ( x < 0.0 ){
    return -3 ;
 }else{ 
    return 3 ;
 }
 return ((x == 0.0) ? 0 : (x < 0.0) ? -1 : 1 );
}


static void 
pg_Sort2Time( char *start, char *end )
{
int i;
FILE *in, *out;
char line[MAXLTH] ;
int max_line=0;
double ftime ;
char *file_line[MAXFLINE];
struct line_node line_c[MAXFLINE] ;
int node,event;
 
   if ( (in=fopen(start,"r"))==NULL ){
      fprintf( stderr, "cannot open file %s \n", start );
      exit(0);
   }
   if ( (out=fopen(end,"w"))==NULL ){
      fprintf( stderr, "cannot open file %s \n", end );
      exit(0);
   }
 
   while(fgets(line, MAXLTH, in)!=NULL){
      if ( max_line >= MAXFLINE ){
         fprintf(stderr,"Warning: increase MAXFLINE \n");
         return ;
      }
      file_line[max_line] = (char *)malloc((strlen(line)+1)*sizeof(char));
      strcpy( file_line[max_line], line );
      line_c[max_line].line = max_line ;
      sscanf( line, "%d %*d %lf %d", &event, &ftime, &node );
      line_c[max_line].ftime = ftime ;
      max_line++;
   }
   printf("doing qsort for %d lines \n", max_line ); 
   fflush(stdout);
   qsort(line_c,max_line,sizeof(struct line_node), 
	(int (*)(const void*,const void*))pg_compare_time );
   printf("finished qsort for %d lines !\n", max_line ); 
   fflush(stdout);
   for ( i = 0; i < max_line; i++ ){
      fprintf(out,"%s", file_line[line_c[i].line] );
   }
     
   fclose(in);
   fclose(out);

	printf("finished Sort2Time!\n");
	fflush(stdout);
}

/*
    Adjency List Stuff
*/

static struct adj_node *adj[MAXP];
 
static struct adj_node *NewAdjNode( int i )
{
struct adj_node *tmp;
 
   tmp = (adj_node *)malloc(sizeof(struct adj_node));
   tmp->node = i;
   tmp->next = NULL;
   return tmp;
}
 
static int
CheckAdj( int i, int j )
{
struct adj_node *tmp;
 
   for (tmp=adj[i];tmp!=NULL;tmp=tmp->next){
       if (tmp->node == j) return 1;
   }
   for(tmp=adj[j];tmp!=NULL;tmp=tmp->next){
       if (tmp->node == i) return 1;
   }
   return 0;
}
 
static void
AddAdjList( int i, int j )
{
struct adj_node *new, *tmp;
 
    if ( 0 <= i && i < MAXP && 0 <= j && j < MAXP ){
       if ( !CheckAdj(i,j) ){
          new = NewAdjNode(j);
          new->next = adj[i];
          adj[i] = new ;
       }
    }
}

static void
CreateDotFile( char *file )
{
FILE *fp;
int i, j;
struct adj_node *tmp;
 
    fp = fopen(file,"w");
    fprintf(fp,"graph G {\n");
    for (i=0;i<MAXP;i++){
       for (tmp=adj[i];tmp!=NULL;tmp=tmp->next){
          j = tmp->node ;
          fprintf(fp,"%d -- %d;\n", i, j);
       }
    }
    fprintf(fp,"}\n");
    fclose(fp);
}
 
static void
clean_adj( void )
{
int i;
struct adj_node *tmp, *trash;
 
   for (i=0;i<MAXP;i++){
      tmp = adj[i];
      while( tmp != NULL ){
         trash = tmp;
         tmp = tmp->next ;
         free(trash);
      }
   }
}

/* 
    Tacyons Removal Stuff
*/

static struct post_msg *post[MAXP] ;
static struct exec_node *PG_PC[MAXP] ;

static void
clean_pc( void )
{
int i;
struct exec_node *tmp1, *trash;

   for (i=0;i<MAXP;i++){
      tmp1 = PG_PC[i];
      while ( tmp1 != NULL ){
         trash = tmp1 ;
         tmp1 = tmp1->next ;
         free(trash);
      }
   }
}

static struct exec_node 
*pg_NewExecNode( double org, double adj, char *line )
{
struct exec_node *tmp ;

	tmp = (struct exec_node *)malloc( sizeof(struct exec_node) );
	if ( tmp == NULL ){
		fprintf(stderr,"Mem error(pg_NewExecNode) \n");
		exit(0);
	}
	tmp->org_time = org ;
	tmp->adj_time = adj ;
	tmp->next = NULL ;
	tmp->send = NULL ;
	tmp->trace_line = (char *)malloc( (strlen(line)+1)*sizeof(char) );
	strcpy(tmp->trace_line, line);
	return tmp ;
}

static void 
pg_add_pc( struct exec_node *count, int node )
{
struct exec_node *tmp ;

	if ( node < 0 || node >= MAXP ) return ;
	if ( PG_PC[node] == NULL ){
		PG_PC[node] = count ;
	}else{
		for (tmp=PG_PC[node];tmp->next != NULL;tmp=tmp->next){}
		tmp->next = count ;
	}
}

static void 
pg_add_post( int tag , int sender, int recv, 
struct exec_node *scounter, int state )
{
struct post_msg *msg, *tmp;

   if ( recv < 0 || recv >= MAXP ) return ;
   msg = (struct post_msg *)malloc( sizeof(struct post_msg) );
   if ( msg == NULL ){
       fprintf(stderr,"Mem error(pg_add_post)\n");
       exit(0);
   }
   msg->state = state;
   msg->counter = scounter;
   msg->sender = sender;
   msg->tag = tag;
   msg->next = NULL;

   if ( post[recv] == NULL ){
      post[recv] = msg;
   }else{
      for (tmp=post[recv]; tmp->next != NULL; tmp = tmp->next){}
      tmp->next = msg;
   }
}

static int 
pg_probe_delete( int tag , int sender, int recv,
struct exec_node **send, int state )
{
struct post_msg *tmp, *prev ;

if ( recv < 0 || recv >= MAXP || post[recv] == NULL ) return 0;
if ( post[recv]->sender == sender && post[recv]->tag == tag 
		&& post[recv]->state == state ){
	tmp = post[recv] ;
	*send = tmp->counter ;
	post[recv] = tmp->next ;
	/* printf("(1) returning %d \n", send ); */
	free(tmp);
	return 1;	
}
for (tmp=post[recv]; tmp != NULL; tmp=tmp->next ){
	if ( tmp->sender == sender && tmp->tag == tag 
		&& tmp->state == state ){
		*send = tmp->counter ;
		prev->next = tmp->next ;
		/* printf("(2) returning %d \n", send ); */
		free(tmp);
		return 1;	
	}
	prev = tmp;
}
return 0;
}

static void 
pg_RemoveTacyons( char *intrace, char *outtrace )
{
FILE *iptr, *optr;
char line[MAXLTH] , rest[MAXLTH];
int record, event, node;
double ftime ;
struct exec_node *exec, *counter, *sender, *recv ;
int lth, tag, other ;
int i, j, k;
int max_node=0;
struct line_node line_c[MAXFLINE] ;
char *file_line[MAXFLINE] ;

   printf("opening %s \n", intrace );
	fflush(stdout);
   if ( (iptr=fopen(intrace,"r"))== NULL ){
      fprintf( stderr, "cannot open file %s \n", intrace );
      exit(0);
   }
   printf("ok ! \n" );
	fflush(stdout);
   printf("opening %s \n", outtrace );
	fflush(stdout);
   if ( (optr=fopen(outtrace,"w"))== NULL ){
      fprintf( stderr, "cannot open file %s \n", intrace );
      exit(0);
   }
   printf("ok ! \n" );
	fflush(stdout);

   for (i=0;i<MAXP;i++){
      PG_PC[i] = NULL ;
      post[i] = NULL ;
      adj[i] = NULL;
   }	

   printf("reading in trace file ... \n" );
   fflush(stdout);
   while(fgets(line, MAXLTH, iptr)!=NULL){
		/*
      printf("reading %s \n", line );
      fflush(stdout);
		*/
      sscanf(line, "%d %d %lf %d" , &record, &event, &ftime, &node );
      /*
      printf("record=%d event=%d ftime=%lf node=%d\n",record,event,ftime,node);
      */
      if ( node > max_node ) max_node = node ;
      counter = pg_NewExecNode( ftime, ftime, line );
      pg_add_pc( counter, node );
      switch(event){
         case NSEND_BEGIN:
         case NSEND:
            if ( record==EVENTENTRY || record==EVENTMARK ){
               sscanf(line, "%*d %*d %*lf %*d %*d %*d %*d %d %d %d\n", 
                             &lth, &tag, &other);
               if ( pg_probe_delete(tag,node,other,&recv,PSORT_RECV) != 1 ) {
                  pg_add_post(tag,node,other,counter,PSORT_SENT);	
               }else{
                  /*
                  printf("(found from a posted recv at %lf) %d -send-> %d \n", 
                          recv->org_time, node, other );
                  */
                  counter->send = recv ;
               }
               if ( node != other ){
                  /* adjency list: node -- other */
                  AddAdjList(node,other);
               }
	         }
            break;
         case NRECVBLOCK:
         case NRECVSTATUS:
         case NRECV:
            if ( record==EVENTEXIT || record==EVENTMARK ){
               sscanf(line, "%*d %*d %*lf %*d %*d %*d %*d %d %d %d\n",
                             &lth, &tag, &other);
               if ( pg_probe_delete(tag,other,node,&sender,PSORT_SENT) != 1 ){
                  pg_add_post(tag,other,node,counter,PSORT_RECV);
               }else{
                  /*
                  printf("(found from a posted send at %lf) %d -send-> %d \n", 
                          sender->org_time, other, node );
                  */
                  sender->send = counter ;
               }
            }
            break ;
         default:
            break;
      }
   }
   for ( node = 0; node <= max_node; node++ ){
      if ( post[node] != NULL ){
        printf("Warning:node %d has outstanding posted messages \n",
                node);
      }
   }
   fflush(stdout);

   CreateDotFile("pganim.dot");

   printf("doing topological sort ... " );
   fflush(stdout);
	/*
   for ( node = 0; node <= max_node; node++ ){
      printf("#");
      fflush(stdout);
      pg_follow_link(PG_PC[node], 0.0 );	 
   }
	*/
	/******************/
	/* REMOVE TACYONS */
	pg_top_sort(max_node,PG_PC);
   printf("OK!\n");
   fflush(stdout);
	
   j=0;
   for ( i = 0; i <= max_node; i++ ){
      for ( exec=PG_PC[i]; exec != NULL; exec=exec->next ){
         if ( j >= MAXFLINE ){
            fprintf(stderr,"Warning: increase MAXFLINE\n");
            clean_pc();
            clean_adj();
            return ;
         }
         file_line[j] = exec->trace_line ;
         sscanf(file_line[j],"%d",&event);
         line_c[j].line = j;
         line_c[j].ftime = exec->adj_time ;
         j++;
      }
   }
	
   printf("doing quick sort ... \n" ); 
   /* sort */
   qsort(line_c,j,sizeof(struct line_node),
	(int (*)(const void*,const void*))pg_compare_time);
   /* write out lines */
   for ( i = 0; i <j ; i++ ){
      sscanf( file_line[line_c[i].line], 
         "%d %d %*lf %[^\n]", &record, &event, rest );
      fprintf(optr, "%d %d %lf %s\n", record, event, line_c[i].ftime, rest );
   }
	
   fclose( iptr );  fclose( optr );
   clean_pc();
   clean_adj();
}


/* 
   TIMING STUFF 
*/


static int
pg_minus_ref( struct timeval *ref, struct timeval delta )
/* ref = ref - delta */
{
struct timeval tmp ;
 
   tmp.tv_sec = tmp.tv_usec = 0;
 
   if ( delta.tv_sec < 0 ){
      /* delta is -ve */
      /* printf("%d: delta -ve %d:%d\n", me, delta.tv_sec, delta.tv_usec );*/
      delta.tv_usec = 1000000 - delta.tv_usec ;
      delta.tv_sec = pg_abs(delta.tv_sec) - 1;
      /* ref = ref + delta */
      if ( ref->tv_usec + delta.tv_usec < 1000000 ){
         ref->tv_sec = ref->tv_sec + delta.tv_sec ;
         ref->tv_usec = ref->tv_usec + delta.tv_usec ;
      }else{
         ref->tv_usec = (ref->tv_usec + delta.tv_usec) - 1000000 ;
         ref->tv_sec = ref->tv_sec + delta.tv_sec + 1 ;
      }
   }else{
      /* delta is +ve */
      /* printf("%d: delta +ve %d:%d\n", me, delta.tv_sec, delta.tv_usec );*/
      /* ref = ref - delta */
      if ( ref->tv_usec < delta.tv_usec ){
         ref->tv_usec = 1000000 + ref->tv_usec - delta.tv_usec ;
         ref->tv_sec = ref->tv_sec - 1  - delta.tv_sec ;
      }else{
         ref->tv_sec = ref->tv_sec - delta.tv_sec ;
         ref->tv_usec = ref->tv_usec - delta.tv_usec ;
      }
   }
}


static int 
send_time( int rtid )
{
struct timeval ref_time ;
struct timeval tmp;
struct timeval rclk, delta ;
int phost ;
int sec, usec;

     /* start time */
     ref_time.tv_sec = PG_SEC ;
     ref_time.tv_usec = PG_USEC ;  
     /* delta = local clk - remote clk (rtid) */
     phost = pvm_tidtohost(rtid); 
     pvm_hostsync(phost,&rclk,&delta) ;
     /* ref_time = ref_time - delta , ie. remote = local - delta */
     pg_minus_ref( &ref_time, delta );
     sec = ref_time.tv_sec; usec = ref_time.tv_usec ;
     /*****/
     pvm_initsend(PvmDataDefault);
     pvm_pkint(&sec,1,1);
     pvm_pkint(&usec,1,1);
     pvm_send(rtid,PG_CMDTAG);
}


/*  
    USER TASK STUFF 
*/

static struct task_node 
*NewTaskNode( char *label, int number )
{
struct task_node *tmp;

    tmp = (struct task_node *)malloc(sizeof(struct task_node));
    if ( tmp == NULL ){
       fprintf(stderr,"Mem Error(NewTaskNode)\n");
       exit(0);
    }
    tmp->label = (char *)malloc((strlen(label)+1)*sizeof(char));
    strcpy(tmp->label,label);
    tmp->number = number ;
    tmp->next = NULL;
    return tmp;
}

static int 
find_task( char *label, int rtid )
{
struct task_node *tmp;
int task = -1;
int update_flag = 0;

   for ( tmp=TaskList; tmp != NULL; tmp=tmp->next ){
      printf("comparing %s and %s \n", tmp->label, label );
      if ( strcmp(tmp->label,label) == 0 ){
          task = tmp->number ;
          break;
      } 
   }
   if ( task == -1 ){
       task = TASK_COUNTER++;
       update_flag = 1;
   }

   /* send task number */
   pvm_initsend(PvmDataDefault);
   pvm_pkint(&task,1,1);
   pvm_send(rtid,PG_CMDTAG);

   if ( update_flag ){
       /* update Task List here */
       tmp = NewTaskNode(label,task); 
       tmp->next = TaskList;
       TaskList = tmp;
   }
   
   return task;
}

static int
write_label_file( char *outfile )
{
struct task_node *tmp;
FILE *fp;

   fp = fopen(outfile,"w");
   for ( tmp=TaskList; tmp!=NULL; tmp=tmp->next ){
       fprintf(fp,"5\t%d\t%s\n", tmp->number, tmp->label ); 
   }
   fclose(fp);
}

static int
assign_me(int tid)
{
int i, me;
int info;

   me = -1;
   for (i=0;i<MAXP;i++){
     if (me2tid[i] == tid){
       me = i;
       break;
     }
   }
   if (me == -1 && pvm_pstat(tid) == PvmOk ){
      /* rank hasn't been assigned to tid yet */
      me2tid[REG_COUNTER] = tid ;
      me = REG_COUNTER;
      ++REG_COUNTER;
      /* flag it to be alive */
      info = pvm_notify(PvmTaskExit,PG_NOTIFYTAG,1,&tid);
      if ( info >= 0 ) EXIT[me] = ALIVE;
   }
   return me ;
}

static tid2me( int tid )
{
int i;
int me = -1;

   for (i=0;i<MAXP;i++){
      if ( me2tid[i] == tid ){
         me = i;
         break;
      }
   }
   return me;
}

static int
check_recv_pk( void )
{
int i;
int flag = 1;   

   /* no one has registered yet */
   if ( REG_COUNTER == 0 ) return 0;

   for (i=0;i<REG_COUNTER;i++){
      printf("%d:received=%d:expected=%d\n",i,PKCNT[i],PKBOUND[i]);
      if ( EXIT[i] != MURDERED ){
         if ( PKCNT[i] != PKBOUND[i] ){
            flag = 0;
            break;
         }
      }
   }
   return flag ;
}

static int
cleanup( void )
{
   /* remove all pg files in /tmp */ 
   system("rm -f /tmp/pg* 2> /dev/null"); 
}

static int
ok_exit( int me, int rtid )
{
   /* I am now dead */
   if ( EXIT[me] == ALIVE ) EXIT[me] = DEAD;

   /* send ack to exiting process */
   pvm_initsend(PvmDataDefault);
   pvm_send(rtid,PG_TRACETAG);
   
   /* */
   ++TRACE_COUNTER;
}
main() 
{
struct task_node *tmp;
char lfile[MAXFLTH];
char buf[TRACEBUFSIZ];
int n;
int i;
char label[100];
int bufid,bytes,tag,rtid;
int serv_cmd;
int tid,me;
struct timeval ref_time ;
FILE *fp;
int ptid;

if ((ptid=pvm_parent()) == PvmNoParent){
    fprintf(stderr,"don't start from command line ...\n");
    exit(0);
}else{
    /* send ack to server starter */
    printf("sending ACK to %x \n", ptid );
    pvm_initsend(PvmDataDefault);
    pvm_send(ptid,PG_CMDTAG); 
    /* wait for parent to exit */
    while(pvm_pstat(ptid) == PvmOk){}
}

cleanup();

sprintf(logname,"/tmp/pg.trf.%d", getpid() );
sprintf(logname2,"/tmp/pg2.trf.%d", getpid() );

while(1){

   init_stuff();

   if ((fp = fopen(logname,"w")) == NULL) {
         printf("pg_exit: can't reopen tracefile %s \n", logname );
         exit(0);
   }
   while(!FINISH){
      if ( pvm_probe(-1,PG_TRACETAG) ){
          printf("tracetag ... \n");
          bufid = pvm_recv(-1,PG_TRACETAG) ;
          pvm_bufinfo(bufid,&bytes,&tag,&rtid);
          me = tid2me(rtid);
          if ( me >= 0 && me < MAXP ){
             pvm_upkint(&n,1,1);
             printf("received %d from %x(%d) \n",n,rtid,me);
             if ( n < 0 ){
                pvm_upkstr(buf);
                fprintf(fp,"%s",buf);
                ++PKCNT[me] ;
             }else{
                /* receive packet count here */
                PKBOUND[me] = n ;
                ok_exit(me,rtid);
             }
          }
      } else if ( pvm_probe(-1,PG_CMDTAG) ){
         /* process label task request */
         bufid = pvm_recv(-1,PG_CMDTAG) ;
         pvm_bufinfo(bufid,&bytes,&tag,&rtid);
         pvm_upkint(&serv_cmd,1,1);
         printf("command: %d \n", serv_cmd );
         switch (serv_cmd){
             case PgTimeWhat:
                 send_time(rtid);
                 break;
             case PgTaskLabel:
                 pvm_upkstr(label);
                 printf("task label = %s \n", label );
                 find_task(label,rtid);
                 break;
             case PgRegister:
                 me = assign_me(rtid);
                 /* reply with registeration number */
                 pvm_initsend(PvmDataDefault);
                 pvm_pkint(&me,1,1);
                 pvm_send(rtid,PG_CMDTAG);
                 if ( ! pg_time_def() ){
                    /* PG_SEC and PG_USEC not defined yet */
                    gettimeofday(&ref_time,NULL); 
                    PG_SEC = ref_time.tv_sec ;
                    PG_USEC = ref_time.tv_usec ;
                 }
                 break;
             case PgTid2Me:
                 pvm_upkint(&tid,1,1);
                 printf("tid = %d \n", tid );
                 me = assign_me(tid);
                 printf("reply = %d \n", me );
                 pvm_initsend(PvmDataDefault);
                 pvm_pkint(&me,1,1);
                 pvm_send(rtid,PG_CMDTAG); 
                 break;
             case PgMe2Tid:
                 pvm_upkint(&me,1,1);
                 printf("me = %d \n", me );
                 tid = -1;
                 if ( 0 <= me && me < MAXP ) tid = me2tid[me] ; 
                 pvm_initsend(PvmDataDefault);
                 pvm_pkint(&tid,1,1);
                 pvm_send(rtid,PG_CMDTAG); 
                 break;
             case PgHalt:
                 pvm_initsend(PvmDataDefault);
                 pvm_send(rtid,PG_CMDTAG);
                 pvm_exit();
                 cleanup();
                 exit(0);
                 break;
             case PgQuery:
                 pvm_initsend(PvmDataDefault);
                 pvm_pkint(&REG_COUNTER,1,1); 
                 for(i=0;i<REG_COUNTER;i++){
                   pvm_pkint(&me2tid[i],1,1);
                 }
                 pvm_pkint(&TASK_COUNTER,1,1); 
                 for ( tmp=TaskList; tmp!=NULL; tmp=tmp->next ){
                    pvm_pkstr(tmp->label);
                 }
                 pvm_send(rtid,PG_CMDTAG);
                 break;
         }
      } else if ( pvm_probe(-1,PG_NOTIFYTAG) ){
         bufid = pvm_recv(-1,PG_NOTIFYTAG) ;
         pvm_upkint(&rtid,1,1);
         me = tid2me(rtid);
         printf("%x(%d) murdered !\n", rtid, me );
         if ( me >= 0 && me < MAXP ){
            /* if I'm alive I must be murdered */
            if ( EXIT[me] == ALIVE ) EXIT[me] = MURDERED;
         }
      } else if (check_tasks(0) == 0){ /* no profiled task */
         if ( check_recv_pk() ) FINISH = 1;
      }
      fflush(stdout);
   } /* while */
   fclose(fp);

   if ( TRACE_COUNTER > 0 ){
      if ((pwd=getcwd(NULL,256)) == NULL ){
          perror("pwd");
          pvm_exit();
          cleanup();
          exit(2);
      }
      printf("pg_serv working directory = %s \n", pwd );
      free(pwd);

      if ( TASK_COUNTER > 0 ){
         /* writes label file for user tasks */
         write_label_file("pgrc");
      }

      printf("sort & remove tacyons -> trace.trf \n");
      fflush(stdout);
      /* sorts according to time */
      pg_Sort2Time( logname, logname2 );
      /* remove tacyons ! */
		printf("removing tacyons ... \n");
		fflush(stdout);
      pg_RemoveTacyons( logname2 , "trace.trf" );
   }

   cleanup();
   fflush(stdout);
}
pvm_exit() ; 
}

