/* TAPE/PVM %W% %G% */

/* Initialization routine of instrumented tasks */

#include "tape_globals.h"
#include "tape_tags.h"
#include "tape_classes.h"
#include "tape.h"

#include "pvm3.h"

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/param.h>
#include <stdio.h>

extern int getuid( void );

#define FILENAMELEN    25
#define CFILE "tape.cfg"
#define LINELEN 81

/* DEFAULTS */

#define DEFAULTBUFSZ   10000
#define DEFAULTMP         20
#define DEFAULTNBPP       15
#define DEFAULTDELAY       2
#define DEFAULTWINSZ       5
#define DEFAULTVERBOSE     0
#define DEFAULTDYNAHOSTS   0
#define DEFAULTMSELECT all_modules | classe_control | classe_msg | \
                       classe_info | classe_syst    | classe_group


/* GLOBALS */

char     *buffer;         /* event buffer */
int      buffer_size;     /* size of buffer in bytes */
int      taille_buf;
int      buffer_index;    /* position in buffer */
int      nb_buffer;       /* number of event buffers */

char     **table_phase;   /* table with phase names */
int      max_phase;       /* max number of open phases */
int      phase_index;     /* next free location in phase table */

int      pvm_actif;

int      base_s;          /* global time basis in secs */

int      control;         /* tid of control task */

unsigned mselect;         /* event selection mask */

int      taskid;          /* tid of this task */

char     *file_name;      /* of trace file */


/* function to compute diff between two timevals */

void tape_clockdiff(tv2, tv1)
     struct timeval *tv2;
     struct timeval *tv1;

     /* tv2>=tv1 => tv2-=tv1 */

{
  tv2->tv_sec-=tv1->tv_sec;
  tv2->tv_usec-=tv1->tv_usec;
  if(tv2->tv_usec<0) {
    tv2->tv_sec--;
    tv2->tv_usec+=1000000;
  }
}


/* function to report errors in mallocs */

void
fatalMallocFailure(fi, ln) 
     char* fi;
     int ln;
{
   fprintf(stderr,"[%s] fatal malloc error in line %d.\n", fi, ln);
   fflush(stdout);
}


/* function to initialize Tape */

void
tape_start(argc, argv, bufsz, nbphas, delay, nbpp)
     int *argc;
     char **argv;
     int bufsz;
     int nbphas;
     int delay;
     int nbpp;
{
  FILE *F;
  int my_argc;
  int i;

  /* init 'taskid' and create 'file_name' */

  taskid = pvm_mytid();
  if(!(file_name=malloc(FILENAMELEN + 1)))
    fatalMallocFailure(__FILE__,__LINE__);
  sprintf(file_name,"/tmp/tape%d", getuid());
  if((F=fopen(file_name,"w"))==NULL)
    fprintf(stderr,"[tape] warning, can't open local trace file %s.\n", file_name);
  else
    fclose(F);

  if( argc==NULL )
    /* we are called from FORTRAN */
    my_argc = 1;
  else
    my_argc = *argc;


  /* I am the master task:
   *
   * - get my operating parameters from command line
   *   or use default values
   * - create tape control task
   */

  if(pvm_parent()<=0) 
    
    {

      /* params for to create tape_control */
    int tape_nbpp;
    int tape_winsz;
    int tape_delay;
    int tape_verbose;
    int tape_dynahosts;

    int err;

    buffer_size   = bufsz;
    if(buffer_size<=0)
      buffer_size = DEFAULTBUFSZ;
    taille_buf    = buffer_size;

    max_phase     = nbphas; 
    if(max_phase<=0) 
      max_phase = DEFAULTMP;

    tape_nbpp  = nbpp;
    if(tape_nbpp<=0) 
      tape_nbpp = DEFAULTNBPP;

    tape_delay = delay;
    if(tape_delay<=0)
      tape_delay = DEFAULTDELAY;

    tape_winsz = DEFAULTWINSZ;

    tape_verbose = DEFAULTVERBOSE;

    tape_dynahosts = DEFAULTDYNAHOSTS;

    mselect = DEFAULTMSELECT; 

    err = tape_params(&tape_nbpp,
		      &tape_winsz,
		      &tape_delay,
		      &tape_verbose,
		      &tape_dynahosts);

    control = tape_create_control( buffer_size,
				   tape_nbpp,
				   tape_delay,
				   tape_winsz,
				   tape_verbose,
				   tape_dynahosts );

    if( control <= 0 ) {
      fprintf(stderr,"[tape] fatal error, can't create control task.\n");
      fflush(stderr);
      exit(1);
    }

    tape_welcome( buffer_size, 
		  max_phase,
		  mselect, 
		  tape_delay, 
		  tape_nbpp, 
		  tape_winsz);

    if(err<0)
      printf("[tape] using default parameters.\n\n");
    else
      printf("[tape] using your parameters in %s (%d errors)\n\n",
	     CFILE,
	     err);

    base_s = tape_ready( control );

  }

  /* I am a slave task:
   *
   * - get my operating parameters from my parent
   */

  else 

    {
      tape_get_params(&buffer_size,
		      &max_phase,
		      &base_s,
		      &control,
		      &mselect);
		      
      taille_buf=buffer_size;
	
    }

  /* inits common to master and slave tasks */

    /* trace buffer */
    
  if ( ! ( buffer = malloc(buffer_size) ) )
    fatalMallocFailure(__FILE__, __LINE__);

  buffer_index = 0;
  buffer[0]=0;
  nb_buffer= 1;

    /* phase name buffer */

  if(!(table_phase=(char **)calloc(sizeof(char *),max_phase)))
    fatalMallocFailure(__FILE__,__LINE__);
  for(i=0;i<max_phase;i++)
    table_phase[i]=NULL;

  phase_index=0;

    /* tape active flag */

  pvm_actif = 1;

}


/*
 * Tape_create_control
 *    Spawn the tape_control task with the
 *    appropriate operating parameters.
 * Return
 *    The tid of the control task ( <=0 on failure )
 */

int
tape_create_control( buffer_size, tape_nbpp, tape_delay, tape_winsz, tape_verbose, tape_dynahosts )
     int buffer_size;
     int tape_nbpp;
     int tape_delay;
     int tape_winsz;
     int tape_verbose;
     int tape_dynahosts;
{
  char *getwd(char*);
  char pathname[MAXPATHLEN];
  int new_buf, old_buf;
  int ctrl;

  /* we leave it up to pvm where to place tape's control task */

  if(pvm_spawn("tape_control",0,PvmTaskDefault,0,1,&ctrl)!=1)
    return(-1);

  /* get current working directory pathname */

  getwd(pathname);

  /* send operating information to tape_control */

  new_buf = pvm_mkbuf(PvmDataDefault);
  old_buf = pvm_setsbuf(new_buf);

  pvm_pkint(&buffer_size,1,1);
  pvm_pkint(&tape_nbpp,1,1);
  pvm_pkint(&tape_delay,1,1);
  pvm_pkint(&tape_winsz,1,1);
  pvm_pkint(&tape_verbose,1,1);
  pvm_pkint(&tape_dynahosts,1,1);
  pvm_pkstr(pathname);
  pvm_send(ctrl,msgtag_taille);

  pvm_setsbuf(old_buf);
  pvm_freebuf(new_buf);

  return( ctrl );
}

/*
 * Tape_ready
 *    Wait for control task to be ready
 * Return
 *    The time basis
 */

int
tape_ready( control )
     int control;
{
  int base;
  int buf, oldbuf;

  oldbuf=pvm_setrbuf(0);

  buf=pvm_recv(control,msgtag_init);

  pvm_upkint(&base,1,1);

  pvm_freebuf(buf);
  pvm_setrbuf(oldbuf);

  return(base);
}

/*
 * Tape_get_params
 *    Get a spwaned task's operat'g parameters from its parent
 * Return
 *    The tasks operat'g parameters (by reference)
 */

void
tape_get_params( bufsz, nbphas, base_s, control, mselect )
     int *bufsz;
     int *nbphas;
     int *base_s;
     int *control;
     unsigned *mselect;
{
  int buf, oldbuf;

  oldbuf=pvm_setrbuf(0);

  buf=pvm_recv(pvm_parent(),msgtag_params);

  pvm_upkint(bufsz,1,1);
  pvm_upkint(nbphas,1,1);
  pvm_upkint(base_s,1,1);
  pvm_upkint(control,1,1);
  pvm_upkuint(mselect,1,1);

  pvm_freebuf(buf);
  pvm_setrbuf(oldbuf);
}

/*
 * Tape_send_params
 *    Send operating parameters to a newly spawned task
 */

void
tape_send_params( who, bufsz, nbphas, base_s, control, mselect )
     int who;
     int bufsz;
     int nbphas;
     int base_s;
     int control;
     unsigned mselect;
{
  int oldbuf, newbuf;

  newbuf = pvm_mkbuf(PvmDataDefault);
  oldbuf = pvm_setsbuf(newbuf);

  pvm_pkint(&bufsz,1,1);
  pvm_pkint(&nbphas,1,1);
  pvm_pkint(&base_s,1,1);
  pvm_pkint(&control,1,1);
  pvm_pkuint(&mselect,1,1);

  pvm_send(who, msgtag_params);

  pvm_freebuf(newbuf);
  pvm_setsbuf(oldbuf);
}

/* tape_notify_control, notify the control task about
 * the number of successfully spawned tasks
 */

void
tape_notify_control(nbtask)
	int nbtask;
{
  int buf, oldbuf;

  buf=pvm_mkbuf(0);
  oldbuf=pvm_setsbuf(buf);
  pvm_pkint(&nbtask,1,1);
  pvm_send(control,msgtag_spawn);
  pvm_freebuf(buf);
  pvm_setsbuf(oldbuf);
}

/* tape_read_op, read current operating parameters
 */

void
tape_read_op(bufsz, nbphas, base, ctrl, binmsk)
	int *bufsz;
	int *nbphas;
	int *base;
	int *ctrl;
	unsigned *binmsk;
{
  *bufsz = buffer_size;
  *nbphas = max_phase;
  *base = base_s;
  *ctrl = control;
  *binmsk = mselect;
}

/* Tape_welcome
 */

void
tape_welcome(bufsz, nbphas, mselect, delay, nbpp, winsz)
     int bufsz;
     int nbphas;
     unsigned mselect;
     int delay;
     int nbpp;
     int winsz;
{
  char mask[33];
  tape_get_mask( mask );
  printf("\n");
  printf("[tape] TAPE/PVM (%s), (c) 1994,95 by LMC-IMAG, Grenoble, France.\n",
         TAPE_VERSION);
  printf("[tape] Runtime parameters : ");
  printf("T=%d ", bufsz);
  printf("P=%d ", nbphas);
  printf("M='%s' ", mask);
  printf("A=%d ", delay);
  printf("E=%d ", nbpp);
  printf("W=%d\n\n", winsz);
}

/* read operation parameters from tape config file
   in current directory

   initializes
     params for tape_control task (passed as args)
     tape globals

   returns

   -1  if no config file found
    0  if config file correctly parsed
    >0 number of errors in parameter assignments
 */

int
tape_params(tape_nbpp,tape_winsz,tape_delay,tape_verbose,tape_dynahosts)
     int *tape_nbpp;
     int *tape_winsz;
     int *tape_delay;
     int *tape_verbose;
     int *tape_dynahosts;
{
  FILE *f;
  char line[LINELEN];
  char pname[10]; /*param name */
  char pval[33]; /* param val */
  int err=0; /* number of errors in params defs */
  int i;

  /* search config file */

  if((f=fopen(CFILE,"r"))==NULL)
    return -1;

  /* parse config file */

  while(!(feof(f))) {

    fgets(line,LINELEN,f);

    /* skip comments */
    if(line[0]=='\n')
      continue;

    i=0; while( line[i]==' ' || line[i]=='\t' )
      i++;

    if(line[i]=='#')
      continue;

    if(sscanf(line,"%s %s",pname,pval)!=2) {
      err++;
      continue;
    }
    
    if(strlen(pname)!=1) {
      err++;
      continue;
    }

    switch(pname[0]) {
      
    case 'T': /* buffer size */
      
      buffer_size = atoi(pval);
      if( buffer_size <= 0 ) 
	{
	  buffer_size = DEFAULTBUFSZ;
	  fprintf(stderr,"[tape] warning, using T=%d.\n",buffer_size);
	}
      taille_buf = buffer_size;
      break;
      
    case 'E': /* number of message exchanges   */
      /* 0 -> no clock synchronization */
      *tape_nbpp = atoi(pval);
      if( *tape_nbpp < 0 )
	{
	  *tape_nbpp = DEFAULTNBPP;
	  fprintf(stderr,"[tape] warning, using E=%d.\n",*tape_nbpp);
	}
      break;
      
    case 'A': /* delay */
      *tape_delay = atoi(pval);
      if( *tape_delay <= 0 )
	{
	  *tape_delay = DEFAULTDELAY;
	  fprintf(stderr,"[tape] warning, using A=%d.\n",*tape_delay);
	}
      break;
      
    case 'M': /* event selection mask */
      if( tape_sm(pval)<0 )
	{
	  mselect = DEFAULTMSELECT;
	  tape_get_mask(pval);
	  fprintf(stderr,"[tape] warning, using M=%s.\n",pval);
	}
      break;
      
    case 'W': /* filter size */
      *tape_winsz = atoi(pval);
      if( *tape_winsz <= 0 )
	{
	  *tape_winsz = DEFAULTWINSZ;
	  fprintf(stderr,"[tape] warning, using W=%d.\n",*tape_winsz);
	}
      break;
      
    case 'P': /* max number of phases */
      max_phase = atoi(pval);
      if( max_phase <=0 )
	{
	  max_phase = DEFAULTMP;
	  fprintf(stderr,"[tape] warning, using P=%d.\n", max_phase);
	}
      break;
      
    case 'V': /* verbose mode */
      *tape_verbose = atoi(pval);
      if( (*tape_verbose<0)&&(*tape_verbose>1) ) {
	*tape_verbose = DEFAULTVERBOSE;
	fprintf(stderr,"[tape] warning, using V=%d\n", *tape_verbose);
      }
      break;
      
    case 'D': /* dyna hosts (dynamic config) */
      *tape_dynahosts = atoi(pval);
      if( (*tape_dynahosts<0)&&(*tape_dynahosts)>1 ) {
	*tape_dynahosts = DEFAULTDYNAHOSTS;
	fprintf(stderr,"[tape] warning, using D=%d\n", *tape_dynahosts);
      }
      break;
      
    default:
      fprintf(stderr,"[tape] warning, unknown param. $s\n", pname);
    }
    
  }
  
  fclose(f);
  
  return err;
}
