/* tape2npicl: convert TAPE/PVM traces to new picl format
 *
 * Author: Eric Maillet, LMC-INPG, Grenoble, France
 * Date  : december 1994
 *
 * Notes : the TAPE/PVM input trace has to be sorted by date
 */

/* $Log$ */

#include "tape_reader.h"
#include "tape_events.h"

#include <stdio.h>

/* maximal number of PICL nodes */

#define MAXNODES      512

/* PICL Record types */

#define RT_MARK       -2
#define RT_ENTRY      -3
#define RT_EXIT       -4

/* PICL Event types */

#define ET_TRACENODE    -901
#define ET_SEND0         -21
#define ET_RECV0_NOWAIT  -51
#define ET_RECV0_WAIT    -52
#define ET_IDLE         -601
#define ET_OVERHEAD     -602
#define ET_USER           99

/* PICL Event Header Generation Macro
 *
 *  rt = Record Type
 *  et = Event Type
 *  nb = Number of Data Fields
 */

#define EH_MACRO(rt,et,nb)                          \
 rt, et,                                            \
 tape_get_s( e->header.date_s, e->header.date_us ), \
 Task2Node( e->header.task ),                       \
 -1,nb

/* Add the delta delay to event date. Used to compute
 * the date of exit events
 */

#define DatePlusDelta                                       \
 e->header.date_s+=e->delta_s;                              \
 e->header.date_s+=(e->header.date_us+e->delta_us)/1000000; \
 e->header.date_us=(e->header.date_us+e->delta_us)%1000000

/* convert PVM Task to PICL node */

int
Task2Node( task )
     int task;
{
  static int n = 0;
  static int NT[MAXNODES];

  int i;

  for(i=0;i<n;i++)
    if(task==NT[i])
      break;

  if( i < n )   /* found task in i */
    return i;

  NT[i] = task; /* insert new task at i */
  n++;

  return i;
}

/* generate idle state on a node */

void
tape_generate_idle_state(node, start_s, delta_s)
     int node;
     double start_s;
     double delta_s;
{
  printf("%d %d %.6f %d %d %d\n",
	 RT_ENTRY, ET_IDLE, start_s, node, -1, 0);
  printf("%d %d %.6f %d %d %d\n",
	 RT_EXIT, ET_IDLE, start_s+delta_s, node, -1, 0);
}

/* generate overhead state on a node */

void
tape_generate_overhead_state(node, start_s, delta_s)
     int node;
     double start_s;
     double delta_s;
{
  printf("%d %d %.6f %d %d %d\n",
	 RT_ENTRY, ET_OVERHEAD, start_s, node, -1, 0);
  printf("%d %d %.6f %d %d %d\n",
	 RT_EXIT, ET_OVERHEAD, start_s+delta_s, node, -1, 0);
}

/* main event converting loop */

void main(argc, argv)
int argc;
char *argv[];
{

  TapeEvent *evt;
  TapeTrace t;
  TapeTaskList tl;

  /* check command line */

  if( argc > 2 ) {
    fprintf(stderr, "usage: %s <tape-trace>\n", argv[0]);
    exit(1);
  }

  /* open a tape trace file */

  switch(argc) {

  case 1:
    t = tape_open_trace( NULL );
    break;
    
  case 2:
    t = tape_open_trace(argv[1]);
    break;
  }

  if( t == NULL ) {
    fprintf(stderr, "tape_reader: cannot open %s\n", argv[1]);
    exit(2);
  }

  /* browse through the trace */

  tape_init_event(&evt);

  while( tape_read_event( t, &evt ) ) 
    {

    switch( evt->header.type ) 
      {
      case event_mytid:
	{
	  TapeMytidEvent *e = (TapeMytidEvent *) evt;
	  printf( "%d %d %.6f %d %d %d\n",
                  EH_MACRO(RT_ENTRY, ET_TRACENODE, 0)
                );
	  break;
	}

      case event_exit:
	{
	  TapeExitEvent *e = (TapeExitEvent *) evt;
	  printf( "%d %d %.6f %d %d %d\n",
                  EH_MACRO(RT_EXIT, ET_TRACENODE, 0)
                );
	  break;
	}

      case event_open_phase:
      case event_close_phase:
	{
	  /* when a phase is opened or closed, we generate an
	     user mark event - this makes Paragraph stop at
	     start and end of phases ( we also put the name of 
	     the phase in the PICL trace although this information
	     is not used by Paragraph ) */

	  TapeOpenPhaseEvent *e = (TapeOpenPhaseEvent *) evt;
	  printf( "%d %d %.6f %d %d %d %d %s\n",
                  EH_MACRO(RT_MARK, ET_USER, 1),1,
		  e->phase
                );
	  break;
	}
	  

      case event_send:
	{
	  TapeSendEvent *e = (TapeSendEvent *) evt;
	  printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
	          EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		  e->bytes,
		  e->msgtag,
		  Task2Node(e->tid)
		);

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		  EH_MACRO(RT_EXIT, ET_SEND0, 0)
		);
	  break;
	}

      case event_recv:
	{
	  int et;
	  TapeRecvEvent *e = (TapeRecvEvent *) evt;

	    /* check if msg arrived */
	  if( e->arrived )
	    et = ET_RECV0_NOWAIT;
	  else
	    et = ET_RECV0_WAIT;

	  printf( "%d %d %.6f %d %d %d %d\n",
	          EH_MACRO(RT_ENTRY, et, 1),2,
		  e->msgtag
		);

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		  EH_MACRO(RT_EXIT, et, 3),2,
		  e->bytes,
		  e->msgtag,
		  Task2Node(e->tid)
		);
	  break;
	}

      case event_mcast:
	{

	  /* MCAST is converted into a series of send events. One
	     of these send events takes account of the overhead
	     inferred by the MCAST call. The other send are null
	     overhead. */

	  TapeMcastEvent *e = (TapeMcastEvent *) evt;
	  int task_count = 0;
	  int k;

	  tl = e->TaskList;

	  while( tl ) /* send entry events */ {
	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( tape_get_next_task( &tl ) )
		   );
	    task_count++;
	  }

	  for( k=0; k < task_count-1; k++ ) /* send exit events */ {
	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );
	  }

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		 EH_MACRO(RT_EXIT, ET_SEND0, 0)
		 );

	  break;
	}

      case event_bcast:
	{

	  /* BCAST same as MCAST except that we get the tasklist
	     by a call to the 'tape_get_tasks_in_group' function */

	  TapeBcastEvent *e = (TapeBcastEvent *) evt;
	  TapeTaskList tl;
	  int task_count = 0;
	  int k;

	  tl = tape_get_tasks_in_group( e->group );

	  while( tl ) /* send entry events */ {
	    printf( "%d %d %.6f %d %d %d %d %d %d %d\n",
		   EH_MACRO(RT_ENTRY, ET_SEND0, 3),2,
		   e->bytes,
		   e->msgtag,
		   Task2Node( tape_get_next_task( &tl ) )
		   );
	    task_count++;
	  }

	  for( k=0; k < task_count-1; k++ ) /* send exit events */ {
	    printf( "%d %d %.6f %d %d %d\n",
		   EH_MACRO(RT_EXIT, ET_SEND0, 0)
		   );
	  }

	  DatePlusDelta;

	  printf( "%d %d %.6f %d %d %d\n",
		 EH_MACRO(RT_EXIT, ET_SEND0, 0)
		 );

	  break;
	}

      case event_barrier:
	{
	  TapeBarrierEvent *e = (TapeBarrierEvent *) evt;

	  tape_generate_idle_state( Task2Node(e->header.task),
				    tape_get_s(e->header.date_s,
					       e->header.date_us),
				    tape_get_s(e->delta_s,
					       e->delta_us)
				   );
	  break;
	}

      case event_trace:
	{
	  TapeTraceEvent *e = (TapeTraceEvent *) evt;

	  tape_generate_overhead_state( Task2Node(e->header.task),
				        tape_get_s(e->header.date_s,
						   e->header.date_us),
				        tape_get_s(e->header.file,
						   e->header.line)
				       );
	  break;
	}

      case  event_pkbyte:
      case  event_pkcplx:
      case  event_pkdcplx:
      case  event_pkdouble:
      case  event_pkfloat:
      case  event_pkint:
      case  event_pklong:
      case  event_pkshort:
      case  event_pkuint:
      case  event_pkulong:
      case  event_pkushort:
      case  event_pkstr:
      case  event_upkbyte:
      case  event_upkcplx:
      case  event_upkdcplx:
      case  event_upkdouble:
      case  event_upkfloat:	
      case  event_upkint:
      case  event_upklong:
      case  event_upkshort:
      case  event_upkuint:
      case  event_upkulong:
      case  event_upkushort:
      case  event_upkstr:
	{
	  /* (U)PK events generate overhead states. Because only the
	     delta fields are needed, all the data structures are mapped
	     onto the Pk1Event */

	  TapePk1Event *e = (TapePk1Event *) evt;
	  tape_generate_overhead_state( Task2Node(e->header.task),
				        tape_get_s(e->header.date_s,
						   e->header.date_us),
				        tape_get_s(e->delta_s,
						   e->delta_us)
				       );
	  break;
	}

      }
  }

  tape_dispose_event( &evt );

  tape_close_trace( t );
}
