/* TAPE/PVM trace reader library */

/* $Log: tape_reader.c,v $
 * Revision 1.2  1994/12/21  10:47:19  maillet
 * Support for multiple open trace files
 * */

#include "tape_reader.h"
#include "tape_events.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAXEVENTLEN 256
#define TOKENSEP    " \n"

/* Macros to extract data fields from events. Arguments are 
 * the name of the event structure followed by the field names
 * of the event structure. The order of the field names in
 * the arg list is important. Special macros are used for group
 * related events ( string type in second place ).
 */

#define MKEVT(Type)                      \
 Type *evt;                              \
 evt = (Type *) malloc(sizeof(Type));    \
 *e = (TapeEvent *) evt;                 \
 tape_read_header(type, buffer, &(evt->header) )

#define FMTEVT1(Type,f1)                 \
 MKEVT(Type);                            \
 evt->f1 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT2(Type,f1,f2)              \
 FMTEVT1(Type,f1);                       \
 evt->f2 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT3(Type,f1,f2,f3)           \
 FMTEVT2(Type,f1,f2);                    \
 evt->f3 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT4(Type,f1,f2,f3,f4)        \
 FMTEVT3(Type,f1,f2,f3);                 \
 evt->f4 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT5(Type,f1,f2,f3,f4,f5)     \
 FMTEVT4(Type,f1,f2,f3,f4);              \
 evt->f5 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT6(Type,f1,f2,f3,f4,f5,f6)  \
 FMTEVT5(Type,f1,f2,f3,f4,f5);           \
 evt->f6 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTEVT7(Type,f1,f2,f3,f4,f5,f6,f7)  \
 FMTEVT6(Type,f1,f2,f3,f4,f5,f6);           \
 evt->f7 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT2(Type,f1,grp)          \
 char *g;                                \
 FMTEVT1(Type,f1);                       \
 g = strchk(strtok(NULL,TOKENSEP));      \
 if(!(evt->grp = malloc(1+strlen(g))))   \
   fprintf(stderr,"FMTGRPEVT2 error\n"); \
 strcpy(evt->grp,g)

#define FMTGRPEVT3(Type,f1,grp,f3)       \
 FMTGRPEVT2(Type,f1,grp);                \
 evt->f3 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT4(Type,f1,grp,f3,f4)    \
 FMTGRPEVT3(Type,f1,grp,f3);             \
 evt->f4 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT5(Type,f1,grp,f3,f4,f5) \
 FMTGRPEVT4(Type,f1,grp,f3,f4);          \
 evt->f5 = atoi(strchk(strtok(NULL,TOKENSEP)))

#define FMTGRPEVT6(Type,f1,grp,f3,f4,f5,f6)    \
 FMTGRPEVT5(Type,f1,grp,f3,f4,f5);             \
 evt->f6 = atoi(strchk(strtok(NULL,TOKENSEP)))

/* globals */

char buffer[MAXEVENTLEN];  /* event read buffer */
int TapeLine;              /* current line number in Tape file */
int TapeError;             /* current library error */
TapeGroupList groups=NULL; /* list of group descriptors */

/* utility functions */

char* strchk( s )
     char *s;
{
  if( s != NULL )
    return s;
  else
    {
      fprintf(stderr,
	      "tape-reader, trace format error in line %d, aborting.\n",
	      TapeLine);
      exit(1);
    }
}

unsigned
tape_get_us(s, us)
     int s;
     int us;
{
  return( s*1000000 + us );
}

double 
tape_get_s( s, us )
     int s;
     int us;
{
  return ( (double) s + 1e-6 * (double) us );
}

/* group and host management functions */

TapeTask
tape_get_next_task(tl)
     TapeTaskList *tl;
{
  TapeTask thisTask;

  assert(*tl != NULL);

  thisTask = (*tl)->task;
  *tl = (*tl)->next;

  return( thisTask );
}

TapeHost
tape_get_next_host(hl)
     TapeHostList *hl;
{
  TapeHost thisHost;

  assert(*hl != NULL);

  thisHost = (*hl)->host;
  *hl = (*hl)->next;

  return( thisHost );
}

void tape_free_tasklist( ll )
     TapeTaskList ll;
{
  while( ll!=NULL ) {
    TapeTaskList tmp;
    tmp = ll->next;
    free(ll);
    ll=tmp;
  }
}

void tape_free_grouplist()
{
TapeGroupList g = groups, gtemp;
TapeTaskList n, ntemp;

while(g!=NULL)
  {
    n=g->tasks;
    while(n!=NULL)
      {
        ntemp=n;
        n=n->next;
        free(ntemp);
      }
    gtemp=g;
    g=g->next;
    free(gtemp);
  }
}


/* GetGroupDesc(groupname)
 *   return group descriptor of groupname
 *   if groupname does not exist a new descr. is created
 */

TapeGroupList
tape_get_group_desc(group)
     char *group;
{
  TapeGroupList g=groups, gl=NULL;

  while(g!=NULL)
    {
      if( strcmp(g->group_name,group) )
	{
	  gl=g;
	  g=g->next;
	}
      else
	break;
    }

  if(g==NULL)
    {
      g=(TapeGroupList)malloc(sizeof(struct s_grouplist));
      strcpy(g->group_name,group);
      g->tasks=NULL;
      g->next=NULL;
      if(gl==NULL)
	groups=g;
      else
	gl->next=g;
    }

  return(g);
}

/* tape_add_task_to_group(groupname,task)
 *   add task to groupname
 */

void
tape_add_task_to_group(group,t)
     char* group;
     TapeTask t;
{
  TapeGroupList g;
  TapeTaskList n, ln=NULL;

  g = tape_get_group_desc(group);

  n=g->tasks;

  while(n!=NULL) {
    ln=n;
    n=n->next;
  }

  n=(TapeTaskList)malloc(sizeof(struct s_tasklist));
  n->next=NULL;
  n->task=t;

  if(ln==NULL)
    g->tasks=n;
  else
    ln->next=n;
}
    

/* tape_del_task_from(groupname,task)
 *   delete a task from groupname
 */

void
tape_del_task_from_group(group,t)
     char* group;
     TapeTask t;
{
  TapeGroupList g;
  TapeTaskList n, ln=NULL;

  g = tape_get_group_desc(group);

  n=g->tasks;

  while(n!=NULL)
    {
      if(n->task!=t)
	{ 
	  ln=n;
	  n=n->next;
	}
      else
	break;
    }

  if(n==NULL)
    fprintf(stderr,"NodeDel, warning, task %x not in group %s\n",
	    t,group);
  else
    {
      if(ln==NULL)
	{
	  g->tasks=n->next;
	  free(n);
	}
      else
	{
	  ln->next=n->next;
	  free(n);
	}
    }
}


TapeTaskList
tape_get_tasks_in_group(group)
     char *group;
{
  TapeGroupList gl = tape_get_group_desc( group );

  return( gl->tasks );
}

/* tracefile management functions */

TapeTrace
tape_open_trace( file )
     char *file;
{
  TapeTrace tt;
  FILE *f;

  TapeLine = 0;
  groups = NULL;

  if( ! ( tt = (TapeTrace) malloc(sizeof(struct s_trace)) ) )
    {
      TapeError = TapeMallocError;
      return( NULL );
    }

  if( file != NULL ) {
    if( (f = fopen(file,"r")) == NULL )
      {
	TapeError = TapeFileError;
	free( (char *) tt );
	return( NULL );
      }
  } else
    f = stdin;

  tt->line = 0;
  tt->file = f;

  TapeError = TapeOK;

  return( tt );
}

void
tape_close_trace( t )
     TapeTrace t;
{
    fclose( t->file );
    free( (char *) t );
    TapeLine=0;
    tape_free_grouplist();
}

/* event management functions */

void
tape_init_event( e )
     TapeEvent **e;
{
  *e = (TapeEvent *) NULL;
}

void
tape_dispose_event( e )
     TapeEvent **e;
{

  if(*e != NULL) {
    switch( (*e)->header.type )
      {
	/* free event lists */
      case event_spawn:
      case event_mcast:
      case event_delhosts:
	tape_free_tasklist(((TapeSpawnEvent *) *e)->TaskList);
	break;
	
	/* free group and phase names */
      case event_barrier:
      case event_getinst:
      case event_bcast:
      case event_gettid:
      case event_gsize:
      case event_joingroup:
      case event_lvgroup:
      case event_open_phase:
      case event_close_phase:
	{
	  TapeGsizeEvent *evt;
	  evt = (TapeGsizeEvent *) *e;
	  free(evt->group);
	}
	break;
      }
    free( *e );
  }
}

int
comment(buf)
     char *buf;
{
  int i=0;
  while(buf[i]==' '||buf[i]=='\t')
    i++;
  
  if(buf[i]=='#'||buf[i]=='\n')
    return 1;
  else
    return 0;
}

void
tape_read_header(type, buf, h)
     int type;
     char *buf;
     TapeEventHeader *h;
{

  h->type = atoi(strchk(strtok(buf,TOKENSEP)));
  h->task = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->file = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->line = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->date_s = atoi(strchk(strtok(NULL,TOKENSEP)));
  h->date_us = atoi(strchk(strtok(NULL,TOKENSEP)));

}

int
tape_read_event( t, e )
     TapeTrace t;
     TapeEvent **e;
{
  FILE *f;
  int type;

  tape_dispose_event( e );

  f = t->file;

  fgets(buffer, MAXEVENTLEN, f);
  t->line++;
  TapeLine=t->line;

  while( !feof(f) && comment(buffer) ) {
    fgets(buffer, MAXEVENTLEN, f);
    t->line++;
    TapeLine=t->line;
  }

  if( feof(f) )
    return 0;

  assert( !feof(f) && !comment(buffer) );

  sscanf(buffer, "%d", &type);

  switch(type) {

  case  event_addhosts:
    {
      FMTEVT2(TapeAddhostsEvent,ret,count);
      break;
    }
    
  case  event_advise:
    {
      FMTEVT2(TapeAdviseEvent,ret,what);
      break;
    }

  case  event_barrier:
    {
      FMTGRPEVT5(TapeBarrierEvent,ret,group,delta_s,delta_us,count);
      break;
    }

  case  event_bcast:
    {
      FMTGRPEVT6(TapeBcastEvent,ret,group,delta_s,delta_us,msgtag,bytes);
      break;
    }

  case  event_bufinfo:
    {
      FMTEVT5(TapeBufinfoEvent,ret,bufid,bytes,msgtag,tid);
      break;
    }

  case  event_config:
    {
      FMTEVT1(TapeConfigEvent,ret);
      break;
    }

  case  event_delete:
    {
      FMTEVT2(TapeDeleteEvent,ret,req);
      break;
    }

  case  event_delhosts:
    {
      TapeHostList queue;
      char *token;

      FMTEVT2(TapeDelhostsEvent,ret,nhost);

      evt->HostList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeHostList tmp;

	  tmp = (TapeHostList) malloc( sizeof(struct s_hostlist) );
	  tmp->next = NULL;
	  tmp->host = atoi(token);

	  if(evt->HostList == NULL) {
	    queue = tmp;
	    evt->HostList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_exit:
    {
      MKEVT(TapeExitEvent);
      break;
    }

  case  event_freebuf:
    {
      FMTEVT2(TapeFreebufEvent,ret,bufid);
      break;
    }

  case  event_getfds:
    {
      FMTEVT1(TapeGetfdsEvent,ret);
      break;
    }

  case  event_getinst:
    {
      FMTGRPEVT3(TapeGetinstEvent,ret,group,tid);
      break;
    }

  case  event_getopt:
    {
      FMTEVT2(TapeGetoptEvent,ret,what);
      break;
    }

  case  event_getrbuf:
    {
      FMTEVT1(TapeGetrbufEvent,ret);
      break;
    }
      
  case  event_getsbuf:
    {
      FMTEVT1(TapeGetsbufEvent,ret);
      break;
    }
      
  case  event_gettid:
    {
      FMTGRPEVT3(TapeGettidEvent,ret,group,inum);
      break;
    }

  case  event_gsize:
    {
      FMTGRPEVT2(TapeGsizeEvent,ret,group);
      break;
    }

  case  event_halt:
    {
      FMTEVT1(TapeHaltEvent,ret);
      break;
    }
      
  case  event_initsend:
    {
      FMTEVT2(TapeInitsendEvent,ret,encoding);
      break;
    }

  case  event_insert:
    {
      FMTEVT3(TapeInsertEvent,ret,req,data);
      break;
    }

  case  event_joingroup:
    {
      FMTGRPEVT2(TapeJoingroupEvent,ret,group);

      if( evt->ret >= 0 )
	tape_add_task_to_group( evt->group, evt->header.task );

      break;
    }

  case  event_kill:
    {
      FMTEVT2(TapeKillEvent,ret,tid);
      break;
    }

  case  event_lookup:
    {
      FMTEVT2(TapeLookupEvent,ret,req);
      break;
    }

  case  event_lvgroup:
    {
      FMTGRPEVT2(TapeLvgroupEvent,ret,group);

      if( evt->ret >=0 )
	tape_del_task_from_group( evt->group, evt->header.task );

      break;
    }

  case  event_mcast:
    {
      TapeTaskList queue;
      char *token;

      FMTEVT6(TapeMcastEvent,ret,delta_s,delta_us,msgtag,bytes,ntask);

      evt->TaskList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeTaskList tmp;

	  tmp = (TapeTaskList) malloc( sizeof(struct s_tasklist) );
	  tmp->next = NULL;
	  tmp->task = atoi(token);

	  if(evt->TaskList == NULL) {
	    queue = tmp;
	    evt->TaskList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_mkbuf:
    {
      FMTEVT2(TapeMkbufEvent,ret,encoding);
      break;
    }

  case  event_mstat:
    /* not supported */
    break;

  case  event_mytid:
    {
      FMTEVT1(TapeMytidEvent,ret);
      break;
    }

  case  event_notify:
    {
      FMTEVT4(TapeNotifyEvent,ret,what,msgtag,ntask);
      break;
    }

  case  event_nrecv:
    {
      FMTEVT6(TapeNrecvEvent,ret,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case  event_parent:
    {
      FMTEVT1(TapeParentEvent,ret);
      break;
    }

  case  event_perror:
    {
      FMTEVT1(TapePerrorEvent,ret);
      break;
    }

  case  event_pkbyte:
  case  event_pkcplx:
  case  event_pkdcplx:
  case  event_pkdouble:
  case  event_pkfloat:
  case  event_pkint:
  case  event_pklong:
  case  event_pkshort:
  case  event_pkuint:
  case  event_pkulong:
  case  event_pkushort:
    {
      FMTEVT5(TapePkEvent,ret,delta_s,delta_us,nitem,stride);
      break;
    }

  /* case  event_packf: */
  case  event_pkstr:
    {
      FMTEVT3(TapePk1Event,ret,delta_s,delta_us);
      break;
    }

  case  event_probe:
    {
      FMTEVT3(TapeProbeEvent,ret,tid,msgtag);
      break;
    }

  case  event_pstat:
    {
      FMTEVT2(TapeKillEvent,ret,tid);
      break;
    }

  case  event_recv:
    {
      FMTEVT7(TapeRecvEvent,ret,arrived,delta_s,delta_us,tid,msgtag,bytes);
      break;
    }

  case  event_recvf:
    {
      MKEVT(TapeRecvfEvent);
      break;
    }

  case  event_send:
    {
      FMTEVT6(TapeSendEvent,ret,tid,delta_s,delta_us,msgtag,bytes);
      break;
    }

  case  event_sendsig:
    {
      FMTEVT1(TapeSendsigEvent,ret);
      break;
    }

  case  event_serror:
    {
      FMTEVT2(TapeSerrorEvent,ret,how);
      break;
    }

  case  event_setdebug:
    {
      FMTEVT2(TapeSetdebugEvent,ret,mask);
      break;
    }

  case  event_setopt:
    /* not supported */
    break;
    
  case  event_setrbuf:
    {
      FMTEVT2(TapeSetrbufEvent,ret,bufid);
      break;
    }

  case  event_setsbuf:
    {
      FMTEVT2(TapeSetsbufEvent,ret,bufid);
      break;
    }

  case  event_spawn:
    {
      TapeTaskList queue;
      char *token;

      FMTEVT3(TapeSpawnEvent,ret,flag,ntask);

      evt->TaskList = NULL;

      while( (token = strtok(NULL,TOKENSEP)) != NULL )
	{
	  TapeTaskList tmp;

	  tmp = (TapeTaskList) malloc( sizeof(struct s_tasklist) );
	  tmp->next = NULL;
	  tmp->task = atoi(token);

	  if(evt->TaskList == NULL) {
	    queue = tmp;
	    evt->TaskList = tmp;
	  }
	  else {
	    queue->next = tmp;
	    queue = tmp;
	  }
	}

      break;
    }

  case  event_start_pvmd:
    {
      FMTEVT2(TapeStart_pvmdEvent,ret,block);
      break;
    }

  case  event_tasks:
    {
      FMTEVT2(TapeTasksEvent,ret,where);
      break;
    }

  case  event_tickle:
    {
      FMTEVT3(TapeTickleEvent,ret, narg, nresp);
      break;
    }

  case  event_tidtohost:
    {
      FMTEVT2(TapeTidtohostEvent,ret, tid);
      break;
    }

  case  event_upkbyte:
  case  event_upkcplx:
  case  event_upkdcplx:
  case  event_upkdouble:
  case  event_upkfloat:	
  case  event_upkint:
  case  event_upklong:
  case  event_upkshort:
  case  event_upkuint:
  case  event_upkulong:
  case  event_upkushort:
    {
      FMTEVT5(TapeUpkEvent,ret,delta_s,delta_us,nitem,stride);
      break;
    }

  /* case  event_unpackf: */
  case  event_upkstr:
    {
      FMTEVT3(TapeUpk1Event,ret,delta_s,delta_us);
      break;
    }

  case  event_version:
    {
      FMTEVT1(TapeVersionEvent,ret);
      break;
    }

  case  event_trace:
    {
      MKEVT(TapeTraceEvent);
      break;
    }

  case  event_open_phase:
    {
      FMTGRPEVT2(TapeOpenPhaseEvent,ret,phase);
      break;
    }

  case  event_close_phase:
    {
      FMTGRPEVT2(TapeClosePhaseEvent,ret,phase);
      break;
    }

  case  event_set_mask:
    {
      FMTGRPEVT2(TapeSetMaskEvent,ret,mask);
      break;
    }

  }

  return 1;

}  
